# API Usage Guide - Roadside Assistance App

## Setup Instructions

1. **Database Setup**
   - Import `roadside_assistance.sql` into your MySQL database
   - Update database credentials in `config.php` if needed

2. **File Structure**
   ```
   /api/
   ├── config.php
   ├── auth.php
   ├── breakdown.php
   └── mechanic.php
   ```

## API Endpoints

### Authentication API (`auth.php`)

#### Register User
```
POST /api/auth.php?action=register
Content-Type: application/json

{
  "name": "John Doe",
  "email": "john@example.com",
  "phone": "+1234567890",
  "password": "password123",
  "role": "driver"
}

// For mechanic registration, add:
{
  "role": "mechanic",
  "garage_name": "Mike's Auto Repair",
  "latitude": -17.8252,
  "longitude": 31.0335,
  "address": "123 Main St, Harare",
  "services": "Flat Tire,Engine Problem,Battery Dead"
}
```

#### Login
```
POST /api/auth.php?action=login
Content-Type: application/json

{
  "email": "john@example.com",
  "password": "password123"
}
```

### Breakdown Management API (`breakdown.php`)

#### Report Breakdown
```
POST /api/breakdown.php?action=report
Content-Type: application/json

{
  "user_id": 1,
  "latitude": -17.8252,
  "longitude": 31.0335,
  "breakdown_type": "Flat Tire",
  "description": "Front left tire is flat",
  "image": "breakdown_image.jpg"
}
```

#### Assign Mechanic
```
POST /api/breakdown.php?action=assign
Content-Type: application/json

{
  "breakdown_id": 1,
  "mechanic_id": 1
}
```

#### Update Status
```
POST /api/breakdown.php?action=update_status
Content-Type: application/json

{
  "breakdown_id": 1,
  "status": "en_route"
}
```

#### Get User History
```
GET /api/breakdown.php?action=user_history&user_id=1
```

#### Get Mechanic Requests
```
GET /api/breakdown.php?action=mechanic_requests&mechanic_id=1
```

#### Get Breakdown Details
```
GET /api/breakdown.php?action=details&breakdown_id=1
```

### Mechanic Discovery API (`mechanic.php`)

#### Find Nearby Mechanics
```
GET /api/mechanic.php?action=nearby&latitude=-17.8252&longitude=31.0335&radius=10&service=Flat Tire
```

#### Get Mechanic Details
```
GET /api/mechanic.php?action=details&mechanic_id=1
```

#### Get All Services
```
GET /api/mechanic.php?action=services
```

#### Update Mechanic Status
```
POST /api/mechanic.php?action=update_status
Content-Type: application/json

{
  "mechanic_id": 1,
  "status": "busy"
}
```

## Response Format

All APIs return JSON responses in this format:

**Success Response:**
```json
{
  "success": true,
  "message": "Operation successful",
  "data": {...}
}
```

**Error Response:**
```json
{
  "error": "Error message here"
}
```

## Status Values

### Breakdown Status
- `pending` - Just reported, waiting for mechanic
- `accepted` - Mechanic assigned and accepted
- `en_route` - Mechanic is on the way
- `completed` - Service completed
- `cancelled` - Request cancelled

### Mechanic Status
- `available` - Ready to take new requests
- `busy` - Currently working on a job
- `offline` - Not available

## Sample Flutter Integration

```dart
// Example API call in Flutter
Future<Map<String, dynamic>> reportBreakdown({
  required int userId,
  required double latitude,
  required double longitude,
  String? breakdownType,
  String? description,
}) async {
  final response = await http.post(
    Uri.parse('$baseUrl/breakdown.php?action=report'),
    headers: {'Content-Type': 'application/json'},
    body: json.encode({
      'user_id': userId,
      'latitude': latitude,
      'longitude': longitude,
      'breakdown_type': breakdownType,
      'description': description,
    }),
  );
  
  return json.decode(response.body);
}
```

## Notes

- All coordinates use decimal degrees format
- Distance calculations use the Haversine formula (results in kilometers)
- Estimated arrival times assume 30 km/h average speed in urban areas
- Image uploads should be handled separately (store filenames in database)
- All APIs include CORS headers for cross-origin requests