<?php
// breakdown.php - Breakdown Management API
require_once 'config.php';
require 'cors.php';

$pdo = getConnection();
$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

switch($method) {
    case 'POST':
        $action = $_GET['action'] ?? '';
        
        if ($action === 'report') {
            reportBreakdown($pdo, $input);
        } elseif ($action === 'assign') {
            assignMechanic($pdo, $input);
        } elseif ($action === 'update_status') {
            updateStatus($pdo, $input);
        } else {
            sendError('Invalid action');
        }
        break;
        
    case 'GET':
        $action = $_GET['action'] ?? '';
        
        if ($action === 'user_history') {
            getUserHistory($pdo);
        } elseif ($action === 'mechanic_requests') {
            getMechanicRequests($pdo);
        } elseif ($action === 'details') {
            getBreakdownDetails($pdo);
        } else {
            sendError('Invalid action');
        }
        break;
        
    default:
        sendError('Method not allowed', 405);
}

function reportBreakdown($pdo, $data) {
    // Validate input
    if (!isset($data['user_id'], $data['latitude'], $data['longitude'])) {
        sendError('Missing required fields');
    }
    
    $userId = $data['user_id'];
    $latitude = $data['latitude'];
    $longitude = $data['longitude'];
    $breakdownType = $data['breakdown_type'] ?? '';
    $description = $data['description'] ?? '';
    $image = $data['image'] ?? '';
    
    try {
        $stmt = $pdo->prepare("INSERT INTO breakdowns (user_id, latitude, longitude, breakdown_type, description, image) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([$userId, $latitude, $longitude, $breakdownType, $description, $image]);
        
        $breakdownId = $pdo->lastInsertId();
        
        sendResponse([
            'success' => true,
            'message' => 'Breakdown reported successfully',
            'breakdown_id' => $breakdownId
        ]);
        
    } catch(PDOException $e) {
        sendError('Failed to report breakdown: ' . $e->getMessage());
    }
}

function assignMechanic($pdo, $data) {
    if (!isset($data['breakdown_id'], $data['mechanic_id'])) {
        sendError('Breakdown ID and Mechanic ID required');
    }
    
    try {
        $stmt = $pdo->prepare("UPDATE breakdowns SET mechanic_id = ?, status = 'accepted' WHERE id = ?");
        $stmt->execute([$data['mechanic_id'], $data['breakdown_id']]);
        
        sendResponse([
            'success' => true,
            'message' => 'Mechanic assigned successfully'
        ]);
        
    } catch(PDOException $e) {
        sendError('Failed to assign mechanic: ' . $e->getMessage());
    }
}

function updateStatus($pdo, $data) {
    if (!isset($data['breakdown_id'], $data['status'])) {
        sendError('Breakdown ID and status required');
    }
    
    $allowedStatuses = ['pending', 'accepted', 'en_route', 'completed', 'cancelled'];
    if (!in_array($data['status'], $allowedStatuses)) {
        sendError('Invalid status');
    }
    
    try {
        $stmt = $pdo->prepare("UPDATE breakdowns SET status = ? WHERE id = ?");
        $stmt->execute([$data['status'], $data['breakdown_id']]);
        
        sendResponse([
            'success' => true,
            'message' => 'Status updated successfully'
        ]);
        
    } catch(PDOException $e) {
        sendError('Failed to update status: ' . $e->getMessage());
    }
}

function getUserHistory($pdo) {
    if (!isset($_GET['user_id'])) {
        sendError('User ID required');
    }
    
    $userId = $_GET['user_id'];
    
    try {
        $stmt = $pdo->prepare("
            SELECT b.*, u.name as mechanic_name, m.garage_name 
            FROM breakdowns b 
            LEFT JOIN mechanics m ON b.mechanic_id = m.id 
            LEFT JOIN users u ON m.user_id = u.id 
            WHERE b.user_id = ? 
            ORDER BY b.created_at DESC
        ");
        $stmt->execute([$userId]);
        $breakdowns = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendResponse([
            'success' => true,
            'breakdowns' => $breakdowns
        ]);
        
    } catch(PDOException $e) {
        sendError('Failed to get history: ' . $e->getMessage());
    }
}

function getMechanicRequests($pdo) {
    if (!isset($_GET['mechanic_id'])) {
        sendError('Mechanic ID required');
    }
    
    $mechanicId = $_GET['mechanic_id'];
    
    try {
        $stmt = $pdo->prepare("
            SELECT b.*, u.name as user_name, u.phone as user_phone 
            FROM breakdowns b 
            JOIN users u ON b.user_id = u.id 
            WHERE b.mechanic_id = ? OR b.status = 'pending'
            ORDER BY b.created_at DESC
        ");
        $stmt->execute([$mechanicId]);
        $requests = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendResponse([
            'success' => true,
            'requests' => $requests
        ]);
        
    } catch(PDOException $e) {
        sendError('Failed to get requests: ' . $e->getMessage());
    }
}

function getBreakdownDetails($pdo) {
    if (!isset($_GET['breakdown_id'])) {
        sendError('Breakdown ID required');
    }
    
    $breakdownId = $_GET['breakdown_id'];
    
    try {
        $stmt = $pdo->prepare("
            SELECT b.*, 
                   u.name as user_name, u.phone as user_phone,
                   mu.name as mechanic_name, m.garage_name, m.latitude as mechanic_lat, m.longitude as mechanic_lng
            FROM breakdowns b 
            JOIN users u ON b.user_id = u.id 
            LEFT JOIN mechanics m ON b.mechanic_id = m.id 
            LEFT JOIN users mu ON m.user_id = mu.id 
            WHERE b.id = ?
        ");
        $stmt->execute([$breakdownId]);
        $breakdown = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($breakdown) {
            sendResponse([
                'success' => true,
                'breakdown' => $breakdown
            ]);
        } else {
            sendError('Breakdown not found', 404);
        }
        
    } catch(PDOException $e) {
        sendError('Failed to get breakdown details: ' . $e->getMessage());
    }
}
?>