<?php
// mechanic.php - Mechanic Discovery API
require_once 'config.php';
require 'cors.php';

$pdo = getConnection();
$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

switch($method) {
    case 'GET':
        $action = $_GET['action'] ?? '';
        
        if ($action === 'nearby') {
            getNearbyMechanics($pdo);
        } elseif ($action === 'details') {
            getMechanicDetails($pdo);
        } elseif ($action === 'services') {
            getServices($pdo);
        } else {
            sendError('Invalid action');
        }
        break;
        
    case 'POST':
        $action = $_GET['action'] ?? '';
        
        if ($action === 'update_status') {
            updateMechanicStatus($pdo, $input);
        } else {
            sendError('Invalid action');
        }
        break;
        
    default:
        sendError('Method not allowed', 405);
}

function getNearbyMechanics($pdo) {
    if (!isset($_GET['latitude'], $_GET['longitude'])) {
        sendError('Latitude and longitude required');
    }
    
    $userLat = $_GET['latitude'];
    $userLng = $_GET['longitude'];
    $radius = $_GET['radius'] ?? 10; // Default 10km radius
    $service = $_GET['service'] ?? ''; // Optional service filter
    
    try {
        $sql = "
            SELECT m.*, u.name, u.phone, u.rating,
                   (6371 * acos(cos(radians(?)) * cos(radians(m.latitude)) * 
                   cos(radians(m.longitude) - radians(?)) + sin(radians(?)) * 
                   sin(radians(m.latitude)))) AS distance
            FROM mechanics m
            JOIN users u ON m.user_id = u.id
            WHERE m.status = 'available'
            HAVING distance < ?
        ";
        
        $params = [$userLat, $userLng, $userLat, $radius];
        
        // Add service filter if specified
        if (!empty($service)) {
            $sql .= " AND m.services LIKE ?";
            $params[] = "%{$service}%";
        }
        
        $sql .= " ORDER BY distance ASC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $mechanics = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate estimated arrival time (assuming 30 km/h average speed)
        foreach ($mechanics as &$mechanic) {
            $mechanic['estimated_arrival'] = round($mechanic['distance'] * 2); // minutes
            $mechanic['services_array'] = !empty($mechanic['services']) ? explode(',', $mechanic['services']) : [];
        }
        
        sendResponse([
            'success' => true,
            'mechanics' => $mechanics,
            'total_found' => count($mechanics)
        ]);
        
    } catch(PDOException $e) {
        sendError('Failed to find nearby mechanics: ' . $e->getMessage());
    }
}

function getMechanicDetails($pdo) {
    if (!isset($_GET['mechanic_id'])) {
        sendError('Mechanic ID required');
    }
    
    $mechanicId = $_GET['mechanic_id'];
    
    try {
        $stmt = $pdo->prepare("
            SELECT m.*, u.name, u.phone, u.email, u.rating
            FROM mechanics m
            JOIN users u ON m.user_id = u.id
            WHERE m.id = ?
        ");
        $stmt->execute([$mechanicId]);
        $mechanic = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($mechanic) {
            $mechanic['services_array'] = !empty($mechanic['services']) ? explode(',', $mechanic['services']) : [];
            
            sendResponse([
                'success' => true,
                'mechanic' => $mechanic
            ]);
        } else {
            sendError('Mechanic not found', 404);
        }
        
    } catch(PDOException $e) {
        sendError('Failed to get mechanic details: ' . $e->getMessage());
    }
}

function getServices($pdo) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM services ORDER BY name");
        $stmt->execute();
        $services = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendResponse([
            'success' => true,
            'services' => $services
        ]);
        
    } catch(PDOException $e) {
        sendError('Failed to get services: ' . $e->getMessage());
    }
}

function updateMechanicStatus($pdo, $data) {
    if (!isset($data['mechanic_id'], $data['status'])) {
        sendError('Mechanic ID and status required');
    }
    
    $allowedStatuses = ['available', 'busy', 'offline'];
    if (!in_array($data['status'], $allowedStatuses)) {
        sendError('Invalid status');
    }
    
    try {
        $stmt = $pdo->prepare("UPDATE mechanics SET status = ? WHERE id = ?");
        $stmt->execute([$data['status'], $data['mechanic_id']]);
        
        sendResponse([
            'success' => true,
            'message' => 'Status updated successfully'
        ]);
        
    } catch(PDOException $e) {
        sendError('Failed to update status: ' . $e->getMessage());
    }
}
?>